// lib/chain.ts
import {
  createWalletClient,
  createPublicClient,
  http,
  encodeFunctionData,
  keccak256,
  toBytes,
  type Hex,
  parseEventLogs,
  getAbiItem,
  getContract,
} from "viem";
import { privateKeyToAccount } from "viem/accounts";
import { foundry } from "viem/chains";
import { TicketRegistryAbi } from "@/lib/abi/TicketRegistry";

// Environment validation
const RPC_URL = process.env.NEXT_PUBLIC_RPC_URL;
export const CONTRACT_ADDRESS = process.env.CONTRACT_ADDRESS as `0x${string}`;
const SERVER_PRIVATE_KEY = process.env.SERVER_PRIVATE_KEY;

// Check if blockchain is properly configured
const isDummyOrMissing = (value: string | undefined) => {
  return (
    !value ||
    value === "0x0000000000000000000000000000000000000000" ||
    value === "dummy" ||
    value === "test" ||
    value.includes("placeholder")
  );
};

export const blockchainEnabled = !!(
  RPC_URL &&
  CONTRACT_ADDRESS &&
  SERVER_PRIVATE_KEY &&
  !isDummyOrMissing(CONTRACT_ADDRESS) &&
  !isDummyOrMissing(SERVER_PRIVATE_KEY)
);

if (!blockchainEnabled) {
  console.warn(
    "⚠️  Blockchain integration disabled - missing or dummy environment variables:"
  );
  console.warn(`   RPC_URL: ${RPC_URL ? "✓" : "✗"}`);
  console.warn(`   CONTRACT_ADDRESS: ${CONTRACT_ADDRESS ? "✓" : "✗"}`);
  console.warn(`   SERVER_PRIVATE_KEY: ${SERVER_PRIVATE_KEY ? "✓" : "✗"}`);
  console.warn("   Using mock blockchain responses");
}

// Create clients only if blockchain is enabled
let account: any = null;
let walletClient: any = null;
let publicClient: any = null;

if (blockchainEnabled) {
  account = privateKeyToAccount(SERVER_PRIVATE_KEY as Hex);

  walletClient = createWalletClient({
    account,
    chain: foundry,
    transport: http(RPC_URL),
  });

  publicClient = createPublicClient({
    chain: foundry,
    transport: http(RPC_URL),
  });
}

export { walletClient, publicClient };

// Build calldata for createTicket
export function encodeCreateTicketArgs(
  jsonHash: Hex,
  severity: number,
  department: string
) {
  return encodeFunctionData({
    abi: TicketRegistryAbi,
    functionName: "createTicket",
    args: [jsonHash, severity, department],
  });
}

// Canonical JSON (stable sort) → keccak256 (bytes32)
export function canonicalJsonHash(obj: unknown): Hex {
  const stable = stableStringify(obj);
  return keccak256(toBytes(stable)) as Hex;
}

function stableStringify(value: any): string {
  if (value === null || typeof value !== "object") return JSON.stringify(value);
  if (Array.isArray(value))
    return "[" + value.map(stableStringify).join(",") + "]";
  const keys = Object.keys(value).sort();
  const entries = keys.map(
    (k) => JSON.stringify(k) + ":" + stableStringify(value[k])
  );
  return "{" + entries.join(",") + "}";
}

// --- Event helpers ---

export const evTicketCreated = getAbiItem({
  abi: TicketRegistryAbi,
  name: "TicketCreated",
});
export const evTicketAssigned = getAbiItem({
  abi: TicketRegistryAbi,
  name: "TicketAssigned",
});
export const evTicketStatus = getAbiItem({
  abi: TicketRegistryAbi,
  name: "TicketStatusChanged",
});

export function parseLogs(logs: any[]) {
  return parseEventLogs({
    abi: TicketRegistryAbi,
    logs,
    strict: false, // ignore unrelated logs
  });
}

// Contract instance used by server (relayer / owner)
export const registry = blockchainEnabled
  ? getContract({
      address: CONTRACT_ADDRESS,
      abi: TicketRegistryAbi,
      client: {
        public: publicClient,
        wallet: walletClient,
      },
    })
  : null;

// Mock blockchain functions for when blockchain is disabled
export const mockBlockchain = {
  async createTicket(jsonHash: Hex, severity: number, department: string) {
    console.log("🔧 Mock: createTicket called", {
      jsonHash,
      severity,
      department,
    });
    return { hash: "0x" + "0".repeat(64) } as any; // Mock transaction hash
  },

  async assignTicket(id: number, assignee: string) {
    console.log("🔧 Mock: assignTicket called", { id, assignee });
    return { hash: "0x" + "0".repeat(64) } as any;
  },

  async updateStatus(id: number, status: number) {
    console.log("🔧 Mock: updateStatus called", { id, status });
    return { hash: "0x" + "0".repeat(64) } as any;
  },

  async getTicket(id: number) {
    console.log("🔧 Mock: getTicket called", { id });
    return {
      hash_: "0x" + "0".repeat(64),
      severity_: 1,
      department_: "Mock Department",
      creator_: "0x" + "0".repeat(40),
      assignee_: "0x" + "0".repeat(40),
      status_: 0,
      createdAt_: Math.floor(Date.now() / 1000),
      updatedAt_: Math.floor(Date.now() / 1000),
    };
  },

  async exists(id: number) {
    console.log("🔧 Mock: exists called", { id });
    return true;
  },

  async totalTickets() {
    console.log("🔧 Mock: totalTickets called");
    return 0;
  },
};

// Safe contract wrapper that uses mock when blockchain is disabled
export const safeRegistry = {
  async createTicket(jsonHash: Hex, severity: number, department: string) {
    if (!blockchainEnabled || !registry || !walletClient) {
      return mockBlockchain.createTicket(jsonHash, severity, department);
    }
    return (registry as any).write.createTicket([
      jsonHash,
      severity,
      department,
    ]);
  },

  async assignTicket(id: number, assignee: `0x${string}`) {
    if (!blockchainEnabled || !registry || !walletClient) {
      return mockBlockchain.assignTicket(id, assignee);
    }
    return (registry as any).write.assignTicket([BigInt(id), assignee]);
  },

  async updateStatus(id: number, status: number) {
    if (!blockchainEnabled || !registry || !walletClient) {
      return mockBlockchain.updateStatus(id, status);
    }
    return (registry as any).write.updateStatus([BigInt(id), status]);
  },

  async getTicket(id: number) {
    if (!blockchainEnabled || !registry || !publicClient) {
      return mockBlockchain.getTicket(id);
    }
    return (registry as any).read.getTicket([BigInt(id)]);
  },

  async exists(id: number) {
    if (!blockchainEnabled || !registry || !publicClient) {
      return mockBlockchain.exists(id);
    }
    return (registry as any).read.exists([BigInt(id)]);
  },

  async totalTickets() {
    if (!blockchainEnabled || !registry || !publicClient) {
      return mockBlockchain.totalTickets();
    }
    return (registry as any).read.totalTickets();
  },
};
